import Foundation

/// Unit tests for Gompertz-Makeham mortality model validation
/// These tests validate the model against known demographic data and research
class GompertzMakehamModelTests {
    
    private let model = GompertzMakehamModel()
    
    /// Test basic mortality rate calculations at various ages
    func testMortalityRateCalculations() -> Bool {
        print("Testing Gompertz-Makeham mortality rate calculations...")
        
        var allTestsPassed = true
        
        // Test key age points with expected approximate mortality rates
        // Based on WHO 2020 mortality data for developed countries
        let testCases: [(age: Double, minRate: Double, maxRate: Double, description: String)] = [
            (25.0, 0.0008, 0.003, "Young adult mortality"),
            (45.0, 0.002, 0.008, "Middle-age mortality"),
            (65.0, 0.008, 0.025, "Early retirement mortality"),
            (75.0, 0.025, 0.065, "Elderly mortality"),
            (85.0, 0.065, 0.150, "Advanced age mortality")
        ]
        
        for testCase in testCases {
            let maleMortality = model.calculateMortality(age: testCase.age, gender: .male)
            let femaleMortality = model.calculateMortality(age: testCase.age, gender: .female)
            
            // Males typically have higher mortality rates
            let maleInRange = maleMortality >= testCase.minRate && maleMortality <= testCase.maxRate
            let femaleInRange = femaleMortality >= testCase.minRate * 0.7 && femaleMortality <= testCase.maxRate * 0.9
            
            print("Age \(testCase.age): Male=\(String(format: "%.6f", maleMortality)), Female=\(String(format: "%.6f", femaleMortality)) - \(testCase.description)")
            
            if !maleInRange || !femaleInRange {
                print("❌ FAILED: \(testCase.description) - rates outside expected range")
                allTestsPassed = false
            } else {
                print("✅ PASSED: \(testCase.description)")
            }
        }
        
        return allTestsPassed
    }
    
    /// Test life expectancy calculations against demographic benchmarks
    func testLifeExpectancyCalculations() -> Bool {
        print("\nTesting life expectancy calculations...")
        
        var allTestsPassed = true
        
        // Test life expectancy at key ages
        // Based on WHO life tables for developed countries (approximate ranges)
        let testCases: [(age: Double, minLE: Double, maxLE: Double, description: String)] = [
            (0.0, 75.0, 85.0, "Life expectancy at birth"),
            (25.0, 52.0, 62.0, "Life expectancy at 25"),
            (45.0, 32.0, 42.0, "Life expectancy at 45"),
            (65.0, 15.0, 25.0, "Life expectancy at 65"),
            (75.0, 8.0, 15.0, "Life expectancy at 75")
        ]
        
        for testCase in testCases {
            let maleLE = model.calculateLifeExpectancy(atAge: testCase.age, gender: .male)
            let femaleLE = model.calculateLifeExpectancy(atAge: testCase.age, gender: .female)
            
            let maleInRange = maleLE >= testCase.minLE && maleLE <= testCase.maxLE
            // Females typically live 4-5 years longer
            let femaleInRange = femaleLE >= testCase.minLE + 3 && femaleLE <= testCase.maxLE + 6
            
            print("Age \(testCase.age): Male=\(String(format: "%.1f", maleLE))y, Female=\(String(format: "%.1f", femaleLE))y - \(testCase.description)")
            
            if !maleInRange || !femaleInRange {
                print("❌ FAILED: \(testCase.description) - life expectancy outside expected range")
                allTestsPassed = false
            } else {
                print("✅ PASSED: \(testCase.description)")
            }
        }
        
        return allTestsPassed
    }
    
    /// Test survival probabilities
    func testSurvivalProbabilities() -> Bool {
        print("\nTesting survival probabilities...")
        
        var allTestsPassed = true
        
        // Test survival from birth to various ages
        let survivalTests: [(toAge: Double, minSurvival: Double, description: String)] = [
            (50.0, 0.95, "Survival to age 50"),
            (70.0, 0.80, "Survival to age 70"),
            (80.0, 0.65, "Survival to age 80"),
            (90.0, 0.25, "Survival to age 90")
        ]
        
        for test in survivalTests {
            let survival = model.calculateSurvivalProbability(toAge: test.toAge, gender: .male)
            
            let testPassed = survival >= test.minSurvival
            print("Survival to \(test.toAge): \(String(format: "%.3f", survival)) - \(test.description)")
            
            if !testPassed {
                print("❌ FAILED: \(test.description) - survival probability too low")
                allTestsPassed = false
            } else {
                print("✅ PASSED: \(test.description)")
            }
        }
        
        return allTestsPassed
    }
    
    /// Test numerical stability for edge cases
    func testNumericalStability() -> Bool {
        print("\nTesting numerical stability...")
        
        var allTestsPassed = true
        
        // Test extreme ages
        let extremeAges = [0.0, 0.1, 120.0, 150.0]
        
        for age in extremeAges {
            let mortality = model.calculateMortality(age: age, gender: .male)
            let lifeExpectancy = model.calculateLifeExpectancy(atAge: age, gender: .male)
            
            let mortalityValid = mortality >= 0.0 && mortality <= 1.0 && !mortality.isNaN
            let lifeExpectancyValid = lifeExpectancy >= 0.0 && !lifeExpectancy.isNaN
            
            print("Age \(age): Mortality=\(String(format: "%.6f", mortality)), LE=\(String(format: "%.2f", lifeExpectancy))")
            
            if !mortalityValid || !lifeExpectancyValid {
                print("❌ FAILED: Numerical instability at extreme age \(age)")
                allTestsPassed = false
            } else {
                print("✅ PASSED: Stable calculations at age \(age)")
            }
        }
        
        return allTestsPassed
    }
    
    /// Test model parameters validation
    func testParameterValidation() -> Bool {
        print("\nTesting parameter validation...")
        
        let isValid = model.validateParameters()
        
        if isValid {
            print("✅ PASSED: All model parameters are within valid ranges")
            
            // Print parameter values for inspection
            let maleParams = model.getParameters(for: .male)
            let femaleParams = model.getParameters(for: .female)
            
            print("Male parameters: A=\(maleParams.A), B=\(maleParams.B), C=\(maleParams.C)")
            print("Female parameters: A=\(femaleParams.A), B=\(femaleParams.B), C=\(femaleParams.C)")
            
            return true
        } else {
            print("❌ FAILED: Model parameters are outside valid ranges")
            return false
        }
    }
    
    /// Run all validation tests
    func runAllTests() -> Bool {
        print("=== Gompertz-Makeham Model Validation Tests ===")
        print("Validating against known mortality tables and demographic research\n")
        
        let tests = [
            testParameterValidation(),
            testNumericalStability(),
            testMortalityRateCalculations(),
            testLifeExpectancyCalculations(),
            testSurvivalProbabilities()
        ]
        
        let allPassed = tests.allSatisfy { $0 }
        
        print("\n=== Test Results ===")
        if allPassed {
            print("🎉 ALL TESTS PASSED - Gompertz-Makeham model is properly implemented")
        } else {
            print("❌ SOME TESTS FAILED - Model requires adjustment")
        }
        
        return allPassed
    }
}

/// Quick validation function for development testing
func validateGompertzMakehamModel() {
    let tests = GompertzMakehamModelTests()
    let _ = tests.runAllTests()
}